/**
 * \file key_container.c
 *
 * \brief Functions for importing key containers
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <fcntl.h>
#include <errno.h>
#include <stdbool.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <libgen.h>

#include <private/sdc_advanced.h>
#include <sdc/daemon/arch/sdc_daemon.h>
#include "key_container.h"
#include "daemon_log.h"
#include "helper.h"

static sdc_error_t key_container_read(int container_fd,
                                      char *container_path, char *filename,
                                      sdc_key_id_t *key_id,
                                      uint8_t **key_secret, size_t *key_secret_len,
                                      daemon_credentials_t *cred)
{
    int err;
    int key_fd;
    uint8_t key[256];
    uint32_t key_len;
    uint32_t key_read = 0;
    struct stat key_stat;
    char *end;
    sdc_error_t sdcerr = SDC_OK;

    /* Convert filename to key id */
    *key_id = strtoul(filename, &end, 10);
    if (*end != '\0') {
        daemon_log(DAEMON_LOG_ERROR,
                   "Error importing key container file: %s/%s is not a valid key-id",
                   container_path, filename);
        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }

    key_fd = openat(container_fd, filename, O_RDONLY);
    if (key_fd < 0) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Error opening key container file %s/%s: %s",
                   container_path, filename, strerror(errno));

        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }


    err = fstat(key_fd, &key_stat);
    if (err) {
        daemon_log(DAEMON_LOG_ERROR, "Unable to determine size of container file(%s/%s): %s",
                   container_path, filename, strerror(errno));
        close(key_fd);
        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }
    key_len = key_stat.st_size;
    if (key_len == 0) {
        daemon_log(DAEMON_LOG_ERROR, "Key container has length zero.");
        close(key_fd);
        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }

    cred->uid = key_stat.st_uid;
    cred->gid = key_stat.st_gid;

    if (key_len > sizeof(key)) {
        daemon_log(DAEMON_LOG_ERROR, "Key container too large");
        close(key_fd);
        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }

    do {
        err = read(key_fd, key, key_len);
        if (err == -1) {
            daemon_log(DAEMON_LOG_ERROR, "Error reading container file");
            close(key_fd);
            sdcerr = SDC_KEY_CONTAINER_IMPORT_FAILED;
            goto out;
        }

        key_read += err;
    } while (err != 0);

    close(key_fd);
    if (key_read != key_len) {
        daemon_log(DAEMON_LOG_ERROR, "Read bytes differ from container file size");
        sdcerr = SDC_KEY_CONTAINER_IMPORT_FAILED;
        goto out;
    }

    /* Install key */
    *key_secret = malloc(key_len);
    if (!(*key_secret)) {
        sdcerr = SDC_NO_MEM;
        goto out;
    }

    *key_secret_len = key_len;
    memcpy(*key_secret, key, key_len);

out:
    keystore_overwrite_secret(key, sizeof(key));

    return sdcerr;
}


/**
 * Import key container
 *
 * \param ctx libkeyman context
 * \param container_path Path to the key container directory
 * \param container_parent Path to the parent of the key container
 *
 * \return 0 on success, -1 otherwise
 */
static sdc_error_t key_container_import(keystore_ctx_t *ctx,
                                        char *container_path,
                                        char *container_parent)
{
    uint8_t *key_secret = NULL;
    size_t key_secret_len = 0;
    sdc_key_id_t key_id;
    sdc_error_t err;
    int fd, dupfd;
    DIR *con_dir = NULL;
    struct dirent *dir_p;
    daemon_credentials_t cred;
    bool import_complete = true;
    int res;

    fd = open(container_path, O_RDONLY);
    if (fd < 0) {
        daemon_log(DAEMON_LOG_ERROR, "Error opening key container path %s: %s", container_path, strerror(errno));
        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }
    /* we need to dup, as fdopendir will use the file descriptor internally */
    dupfd = dup(fd);
    con_dir = fdopendir(dupfd);
    if (con_dir == NULL) {
        daemon_log(DAEMON_LOG_ERROR, "Error opening key container directory %s: %s", container_path, strerror(errno));
        close(fd);
        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }

    /* Check if there is a file in the directory */
    errno = 0;
    while ((dir_p = readdir(con_dir)) != NULL) {
        char *filename = dir_p->d_name;

        /* Skip "." and ".." (also all hidden files) */
        if (filename[0] == '.') {
            continue;
        }

        key_secret = NULL;
        err = key_container_read(fd, container_path, filename, &key_id,
                                 &key_secret, &key_secret_len, &cred);
        if (err == SDC_OK) {
            err = keystore_install_product_key(ctx, key_secret, key_secret_len, &cred, key_id);
            if ((err == SDC_OK) || (err == SDC_KID_EXISTS)) {
                if (err == SDC_KID_EXISTS) {
                    daemon_log(DAEMON_LOG_WARNING,
                               "Key with id %d already existing - skipping this container",
                               key_id);
                }

                /* Key has been imported, so key container file can be deleted */
                res = unlinkat(fd, filename, 0);
                if (res != 0) {
                    daemon_log(DAEMON_LOG_ERROR, "Failed to unlink key container %s/%s: %s",
                               container_path, filename, strerror(errno));
                    import_complete = false;
                }
            } else {
                daemon_log(DAEMON_LOG_ERROR, "Error installing key container %s/%s: %s",
                           container_path, filename, sdc_get_error_string(err) );
                import_complete = false;
            }
        } else {
            import_complete = false;
        }
        if (key_secret != NULL)
            keystore_overwrite_secret(key_secret, key_secret_len);
        free(key_secret);
        key_secret = NULL;
        key_secret_len = 0;
        errno = 0;
    }
    if (errno != 0) {
        daemon_log(DAEMON_LOG_ERROR, "Readdir of key container %s failed: %s",
                   container_path, strerror(errno) );
        import_complete = false;
    }

    /* We don't need con_dir and file descriptor of container any longer  */
    if (closedir(con_dir)) {
        daemon_log(DAEMON_LOG_ERROR, "Failed to close key container directory %s: %s",
                   container_path, strerror(errno));
        import_complete = false;
    }
    if (close(fd)) {
        daemon_log(DAEMON_LOG_ERROR, "Failed to close key container path %s - %d: %s",
                   container_path, fd, strerror(errno));
        import_complete = false;
    }

    if (import_complete) {
        if (rmdir(container_path)) {
            daemon_log(DAEMON_LOG_ERROR, "Failed to remove key container %s: %s",
                       container_path, strerror(errno));
            return SDC_KEY_CONTAINER_IMPORT_FAILED;
        }

        fd = open(container_parent, O_RDONLY);
        if (fd < 0) {
            daemon_log(DAEMON_LOG_ERROR, "Failed to open parent of key container %s for sync: %s",
                       container_parent, strerror(errno));
            return SDC_KEY_CONTAINER_IMPORT_FAILED;
        }

        if (fsync(fd)) {
            daemon_log(DAEMON_LOG_ERROR, "Failed to sync parent of key container %s: %s",
                       container_parent, strerror(errno));
            close(fd);
            return SDC_KEY_CONTAINER_IMPORT_FAILED;
        }

        if (close(fd)) {
            daemon_log(DAEMON_LOG_ERROR, "Failed to close key container parent %s: %s",
                       container_parent, strerror(errno));
            return SDC_KEY_CONTAINER_IMPORT_FAILED;
        }
    } else {
        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }

    return SDC_OK;
}

/**
 * Perform security checks on the key container
 * (e.g. to prevent users are not able to trigger another key container
 * import after the first one has been completed)
 * Furthermore this function needs to securely determine if the
 * key import is supposed to take place - key container path exists
 *
 * Below we use fstat, openat ... alot instead of working on names
 * This is to avoid that a clever attacker (not being root or daemon-user)
 * having write access to some parent directory can fool us.
 * E.g. Remove key_container - parent folder after credentials checking
 * and replace by another folder containing "key_container" directory before
 * checking for key container to trigger another import
 * On the other side we don't want to imply any not required credentials
 * for parent folders of the key_container and its direct parent
 * Please note: Altering the initial key import (e.g. not importing or
 * only importing some key containers) is not considered - This initial
 * key import has to take place during the first system startup
 * (i.e. during production in secure environment)
 * After securely detecting the presence of a key-container (i.e. first time bootup)
 * the daemon is assumed to operate in a secure environment
 *
 * \param container_path Path to the key container directory (complete)
 * \param container_name Just the name of the container entry below its parent
 * \param container_parent Path to the parent of the container directory
 * \param container_exists - return true if container exists and shall be imported, else false
 *
 * \return SDC_OK on success, else corresponding error code
 */
static sdc_error_t key_container_security_checks(char *container_path,
                                                 char *container_name,
                                                 char *container_parent,
                                                 bool *container_exists)
{
    uid_t daemon_uid;
    int parent_fd;
    int container_fd;
    int err;
    struct stat parent_stat;
    struct stat container_stat;

    *container_exists = false;

    /* we are using real uid for all checks */
    daemon_uid = getuid();

    /* check parent folder of key container */
    parent_fd = open(container_parent, O_RDONLY);
    if (parent_fd < 0) {
        if (errno == ENOENT) {
            /* no container parent folder present - skip importing */
            return SDC_OK;
        }

        daemon_log(DAEMON_LOG_ERROR, "Can't access the parent folder of the key-container %s:%s",
                   container_parent,
                   strerror(errno));
        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }

    /* daemon has to be the user
     * it has to be a directory
     * only daemon (==user) (and root) may write to the directory
     *
     * Note:
     * It is not necessary to check that daemon has rx for directory as
     * openat for container would fail later with error different than
     *      ENOENT (container directory not present)
     * Checking of write permissions is only necessary if container
     * directory exists
     */
    if (fstat(parent_fd, &parent_stat)) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Fstat failed for key container parent path %s:%s",
                   container_parent,
                   strerror(errno));
        close(parent_fd);
        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }
    if (daemon_uid != parent_stat.st_uid) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Daemon doesn't own key container parent path %s",
                   container_parent);
        close(parent_fd);
        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }
    if (!S_ISDIR(parent_stat.st_mode)) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Key container parent path %s is no directory",
                   container_parent);
        close(parent_fd);
        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }
    if (parent_stat.st_mode & (S_IWGRP | S_IWOTH)) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Invalid mode (0%o) of key container parent path %s - "
                   "Others or group is allowed to create/delete directory entries.",
                   parent_stat.st_mode & (S_IRWXU | S_IRWXG | S_IRWXO | S_ISVTX | S_ISUID | S_ISGID),
                   container_parent);
        close(parent_fd);
        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }


    /* try to access the container - relative to its parent
     * intended to prevent potential attacks by manipulating
     * directories above the parent during these checks
     */
    container_fd = openat(parent_fd, container_name, O_RDONLY);

    /* this is no longer needed - cleanup */
    if (close(parent_fd)) {
        daemon_log(DAEMON_LOG_WARNING,
                   "Failed to close key container directory %s: %s",
                   container_path, strerror(errno));
    }

    if (container_fd < 0) {
        err = errno;

        if (err == ENOENT) {
            /* no container present - skip importing */
            return SDC_OK;
        }

        daemon_log(DAEMON_LOG_ERROR,
                   "Accessing key container directory %s failed: %s",
                   container_path, strerror(err));
        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }

    /* after this point we can assume that daemon is running in secure environment
     * the checks below are not needed for security but provide the the user
     * with better error messages
     */
    *container_exists = true;
    close(container_fd);

    /* check that daemon has w for parent directory - can remove container after import
     * we need to use access here - as we might be running readonly filesystem
     */
    if (access(container_parent, W_OK | X_OK)) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Daemon has no permissions to delete key container path %s after import.",
                   container_path);
        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }

    /* verify that daemon can access and delete the key containers */
    if (access(container_path, R_OK | W_OK | X_OK)) {
        daemon_log(DAEMON_LOG_ERROR, "Daemon can't rwx key container path %s",
                   container_path);
        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }

    /* check that key container is really a directory (not a link)
     * check that sticky bit is not set for key container path
     * otherwise we might not be able to delete the containers
     *
     * Links are forbidden as the daemon would not do any further import
     * after removing the link, but the real directory persists and some
     * user/component might assume the import is not yet finished
     */
    if (lstat(container_path, &container_stat)) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Lstat failed for key container path %s: %s",
                   container_path, strerror(errno));
        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }
    if (!S_ISDIR(container_stat.st_mode)) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Key container path %s is no directory",
                   container_path);
        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }
    if (container_stat.st_mode & (S_ISVTX)) {
        daemon_log(DAEMON_LOG_ERROR,
                   "Invalid mode (0%o) of key container parent path %s. "
                   "Sticky bit set",
                   container_stat.st_mode & (S_IRWXU | S_IRWXG | S_IRWXO | S_ISVTX | S_ISUID | S_ISGID),
                   container_path);
        return SDC_KEY_CONTAINER_IMPORT_FAILED;
    }

    return SDC_OK;
}

/**
 * Check if key containers shall be imported and import
 *
 * In case the folder does not exist or no key container is configured
 * importing is skipped without any further checks
 *
 * Before performing the actual import this function will perform
 * access permission checks to maintain security (e.g. users are not able
 * to trigger another key container import after the first one has been
 * completed)
 *
 * \param ctx libkeyman context
 *
 * \return SDC_OK on success, else corresponding error code
 */
sdc_error_t key_container_check_and_import(keystore_ctx_t *ctx)
{
    sdc_error_t err;
    char *path;
    char *path_parent_copy;
    char *container_parent;
    char *path_name_copy;
    char *container_name;
    bool do_import = false;


    /* Check if there is a key container file for initial key import */
    err = sdc_config_file_lookup_absolute_path(&path, SDC_CONFIG_CONTAINER);
    if (err == SDC_CONFIG_MISSING) {
        /* no key containers configured - skipping import */
        return SDC_OK;
    }
    if (err != SDC_OK) {
        daemon_log(DAEMON_LOG_ERROR, "Invalid key container config\n");
        return err;
    }

    /* prepare names of parent dir and container
     * we need a copy as dirname, basename may alter the string */
    path_parent_copy = strdup(path);
    if (!path_parent_copy) {
        free(path);
        return SDC_NO_MEM;
    }
    path_name_copy = strdup(path);
    if (!path_name_copy) {
        free(path_parent_copy);
        free(path);
        return SDC_NO_MEM;
    }

    container_parent = dirname(path_parent_copy);
    container_name = basename(path_name_copy);

    err = key_container_security_checks(path, container_name, container_parent, &do_import);

    if ((err == SDC_OK) && (do_import)) {
        /* we can assume that daemon is running in secure environment during import */
        err = key_container_import(ctx, path, container_parent);
    }

    free(path_parent_copy);
    free(path_name_copy);
    free(path);
    return err;
}
